/***************************************
 * copyright (c) Vanden Berghen Frank  *
 * V 1.0                               *
 * *************************************/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <string.h> // pour avoir memmove (microsoft bug)
#include "image.h"
// #include <crtdbg.h>

void imageD::save(char *filename,int i)
{
	image *im3=new image(this,i);
	im3->save(filename); 
	delete im3;
};

void fourn(double data[], unsigned long nn[], int ndim, int isign)
/*Replaces data by its ndim-dimensional discrete Fourier transform, if isign is input as 1.
nn[1..ndim] is an integer array containing the lengths of each dimension (number of complex
values), which MUST all be powers of 2. data is a real array of length twice the product of
these lengths, in which the data are stored as in a multidimensional complex array: real and
imaginary parts of each element are in consecutive locations, and the rightmost index of the
array increases most rapidly as one proceeds along data. For a two-dimensional array, this is
equivalent to storing the array by rows. If isign is input as - 1, data is replaced by its 
inverse transform times the product of the lengths of all dimensions.*/
{
	int idim;
	unsigned long i1,i2,i3,i2rev,i3rev,ip1,ip2,ip3,ifp1,ifp2;
	unsigned long ibit,k1,k2,n,nprev,nrem,ntot;
	double tempi,tempr;
	double theta,wi,wpi,wpr,wr,wtemp; //Double precision for trigonometric recurrences.
	for (ntot=1,idim=1;idim<=ndim;idim++) //Compute total number of complex values
		ntot *= nn[idim];
	nprev=1;
	for (idim=ndim;idim>=1;idim--) 
	{									//Main loop over the dimensions.
		n=nn[idim];
		nrem=ntot/(n*nprev);
		ip1=nprev << 1;
		ip2=ip1*n;
		ip3=ip2*nrem;
		i2rev=1;
		for (i2=1;i2<=ip2;i2+=ip1) 
		{				// This is the bit-reversal section of the routine
			if (i2 < i2rev) 
			{
				for (i1=i2;i1<=i2+ip1-2;i1+=2) 
				{
					for (i3=i1;i3<=ip3;i3+=ip2) 
					{
						i3rev=i2rev+i3-i2;
						SWAP(data[i3],data[i3rev]);
						SWAP(data[i3+1],data[i3rev+1]);
					}
				}
			}
			ibit=ip2 >> 1;
			while (ibit >= ip1 && i2rev > ibit) 
			{
				i2rev -= ibit;
				ibit >>= 1;
			}
			i2rev += ibit;
		}
		ifp1=ip1; //Here begins the Danielson-Lanczos section of the routine. 
		while (ifp1 < ip2) 
		{
			ifp2=ifp1 << 1;
			theta=isign*6.28318530717959/(ifp2/ip1); //Initialize for the trig. recurrnce
			wtemp=sin(0.5*theta);
			wpr = -2.0*wtemp*wtemp;
			wpi=sin(theta);
			wr=1.0;
			wi=0.0;
			for (i3=1;i3<=ifp1;i3+=ip1) 
			{
				for (i1=i3;i1<=i3+ip1-2;i1+=2) 
				{
					for (i2=i1;i2<=ip3;i2+=ifp2) 
					{
						k1=i2; // Danielson-Lanczos formula:
						k2=k1+ifp1;
						tempr=(float)wr*data[k2]-(float)wi*data[k2+1];
						tempi=(float)wr*data[k2+1]+(float)wi*data[k2];
						data[k2]=data[k1]-tempr;
						data[k2+1]=data[k1+1]-tempi;
						data[k1] += tempr;
						data[k1+1] += tempi;
					}
				}
				wr=(wtemp=wr)*wpr-wi*wpi+wr; //Trigonometric recurrence.
				wi=wi*wpr+wtemp*wpi+wi;
			}
			ifp1=ifp2;
		}
		nprev *= n;
	}
}

void rlft3(double ***data, double **speq, unsigned long nn1, unsigned long nn2,
		   unsigned long nn3, int isign)
/*Given a three-dimensional real array data[1..nn1][1..nn2][1..nn3] (where nn1 = 1for
the case of a logically two-dimensional array), this routine returns (for isign=1) the complex
fast Fourier transform as two complex arrays: On output, data contains the zero and positive
frequency values of the third frequency component, while speq[1..nn1][1..2*nn2]contains
the Nyquist critical frequency values of the third frequency component. First (and second)
frequency components are stored for zero, positive, and negative frequencies, in standard wrap-
around order. See text for description of how complex values are arranged. For isign=-1, the
inverse transform (times nn1*nn2*nn3/2 as a constant multiplicative factor) is performed,
with output data (viewed as a real array) deriving from input data (viewed as complex) and
speq. For inverse transforms on data not generated first by a forward transform, make sure
the complex input data array satisfies property (12.5.2). The dimensions nn1, nn2, nn3 must
always be integer powers of 2.*/
{
	void fourn(double data[], unsigned long nn[], int ndim, int isign);
	void nrerror(char error_text[]);
	unsigned long i1,i2,i3,j1,j2,j3,nn[4],ii3;
	double theta,wi,wpi,wpr,wr,wtemp;
	double c1,c2,h1r,h1i,h2r,h2i;
	if ((unsigned long)(1+&data[nn1][nn2][nn3]-&data[1][1][1])!= nn1*nn2*nn3)
	{
		printf("rlft3: problem with dimensions or contiguity of data array\n"); 
		exit(255);
	};
	c1=0.5;
	c2 = -0.5*isign;
	theta=isign*(6.28318530717959/nn3);
	wtemp=sin(0.5*theta);
	wpr = -2.0*wtemp*wtemp;
	wpi=sin(theta);
	nn[1]=nn1;
	nn[2]=nn2;
	nn[3]=nn3 >> 1;
	if (isign == 1) 
	{					// Case of forward transform.
		fourn(&data[1][1][1]-1,nn,3,isign); // Here is where most all of the com-
											// pute time is spent. 
		for (i1=1;i1<=nn1;i1++)
			for (i2=1,j2=0;i2<=nn2;i2++) 
			{					//	Extend data periodically into speq.
				speq[i1][++j2]=data[i1][i2][1];
				speq[i1][++j2]=data[i1][i2][2];
			}
	}
	for (i1=1;i1<=nn1;i1++) 
	{
		j1=(i1 != 1 ? nn1-i1+2 : 1);
		//	Zero frequency is its own reflection, otherwise locate corresponding negative 
		// frequency in wrap-around order.
		wr=1.0; // Initialize trigonometric recurrence.
		wi=0.0;
		for (ii3=1,i3=1;i3<=(nn3>>2)+1;i3++,ii3+=2) 
		{
			for (i2=1;i2<=nn2;i2++) 
			{
				if (i3 == 1) 
				{ 			// Equation (12.3.5).
					j2=(i2 != 1 ? ((nn2-i2)<<1)+3 : 1);
					h1r=c1*(data[i1][i2][1]+speq[j1][j2]);
					h1i=c1*(data[i1][i2][2]-speq[j1][j2+1]);
					h2i=c2*(data[i1][i2][1]-speq[j1][j2]);
					h2r= -c2*(data[i1][i2][2]+speq[j1][j2+1]);
					data[i1][i2][1]=h1r+h2r;
					data[i1][i2][2]=h1i+h2i;
					speq[j1][j2]=h1r-h2r;
					speq[j1][j2+1]=h2i-h1i;
				} else 
				{
					j2=(i2 != 1 ? nn2-i2+2 : 1);
					j3=nn3+3-(i3<<1);
					h1r=c1*(data[i1][i2][ii3]+data[j1][j2][j3]);
					h1i=c1*(data[i1][i2][ii3+1]-data[j1][j2][j3+1]);
					h2i=c2*(data[i1][i2][ii3]-data[j1][j2][j3]);
					h2r= -c2*(data[i1][i2][ii3+1]+data[j1][j2][j3+1]);
					data[i1][i2][ii3]=h1r+wr*h2r-wi*h2i;
					data[i1][i2][ii3+1]=h1i+wr*h2i+wi*h2r;
					data[j1][j2][j3]=h1r-wr*h2r+wi*h2i;
					data[j1][j2][j3+1]= -h1i+wr*h2i+wi*h2r;
				}
			}
			wr=(wtemp=wr)*wpr-wi*wpi+wr; // Do the recurrence.
			wi=wi*wpr+wtemp*wpi+wi;
		}
	}
	if (isign == -1) // Case of reverse transform.
		fourn(&data[1][1][1]-1,nn,3,isign);
};

void imageD::correlation(imageD *im)
{

	// the 2 images must be the same size.
	// if one of the image is smaller than the other. You must pad 0 to
	// enlarge it.
	// Where is THIS in IM ?

	double fac,r,i,*sp1,*sp2;
	DWORD j;

	if ((fft==NULL)||(im->fft==NULL)) 
	{
		printf("no fft calculated.\n"); exit(255);
	};

	fac=2.0/(Height*Width); // Factor needed to get normalized inverse.
	sp1 = fft[1]+1;
	sp2 = im->fft[1]+1;
	for (j=0;j<(DWORD)(Height*Width/2);j++) 
	{
		r =  sp1[0]*sp2[0] + sp1[1]*sp2[1];
		i =  sp1[0]*sp2[1] - sp1[1]*sp2[0];
		sp1[0] = fac*r;
		sp1[1] = fac*i;
		sp1 += 2;
		sp2 += 2;
	}
	sp1 = speq+1;
	sp2 = im->speq+1;
	for (j=0;j<(DWORD)Height;j++) 
	{
		r =  sp1[0]*sp2[0] + sp1[1]*sp2[1];
		i =  sp1[0]*sp2[1] - sp1[1]*sp2[0];
		sp1[0] = fac*r;
		sp1[1] = fac*i;
		sp1 += 2;
		sp2 += 2;
	}

	memcpy(gl[1]+1,fft[1]+1,Size*sizeof(double));
	rlft3(&gl-1,&speq-1,1,Height,Width,-1); // Inverse FFT the product of the two FFTs.
};

void imageD::calc_FFT()
{
	double *tmp;
	int i;
	if (fft==NULL)
	{
		tmp=(double*)malloc(Size*sizeof(double));
		fft=(double**)malloc((Height+1)*sizeof(double*));
		speq=(double*)malloc(1+(2*Height+1)*sizeof(double));
		if ((tmp==NULL)||(fft==NULL)||(speq==NULL))
		{
            fprintf(stderr,"imgD_caclFFT: out of memory.\n"); exit(220);
		};
		tmp--; for (i=1;i<=Height;i++, tmp+=Width) fft[i]=tmp;
		
	};
	memcpy(fft[1]+1,gl[1]+1,Size*sizeof(double));
	rlft3(&fft-1,&speq-1,1,Height,Width,1);
};

imageD::imageD(imageD *im,WORD w,WORD h)
{
	int i;
	double *indexS,*indexD,*tmp,*startD,*indexE;

	Width=MAX(w,im->Width);
	Height=MAX(h,im->Height);
	Size=Width*Height;
		
	startD=tmp=(double*)malloc(Size*sizeof(double));
	gl=(double**)malloc((Height+1)*sizeof(double*));
	if ((tmp==NULL)||(gl==NULL))
	{
        fprintf(stderr,"imgD_enlarge: out of memory.\n"); exit(221);
	};
	tmp--; for (i=1;i<=Height;i++, tmp+=Width) gl[i]=tmp;
	fft=NULL;

	indexS=im->gl[1]+1; indexD=startD; indexE=indexS+im->Size;
	while (indexS!=indexE)
	{
		memcpy(indexD,indexS,im->Width*sizeof(double));
		memset(indexD+im->Width,0,(Width-im->Width)*sizeof(double));
		indexS+=im->Width;
		indexD+=Width;
	}
	memset(indexD,0,Width*(Height-im->Height)*sizeof(double));
};

imageD::imageD(image *im)
{
	double *tmp;
	DWORD i;
	Height=im->Height;
	Width=im->Width;
	Size=im->Size;
	tmp=(double*)malloc(Size*sizeof(double));
    gl=(double**)malloc((Height+1)*sizeof(double*));
 	if ((tmp==NULL)||(gl==NULL))
	{
        fprintf(stderr,"imgD_convert_from_byte: out of memory.\n"); exit(221);
	};
	for (i=0;i<Size;i++) tmp[i]=(double)im->gl[i]-128;	
	tmp--; for (i=1;i<=Height;i++, tmp+=Width) gl[i]=tmp;
	fft=NULL;
};

imageD::imageD(imageD *im,double scaleX,double scaleY)
{
	WORD i,j;
	DWORD incX,incY,counterY,counterX,a;
	double *indexD,*indexS,*indexSstartline,*indexDstartline,*tmp;
	double *surface,*surfaceStartline,*surfaceStartimage;
	double b;

	Width=(WORD)ceil(im->Width*scaleX);
	Height=(WORD)ceil(im->Height*scaleY);
	Size=Width*Height;
	tmp=(double*)malloc(Size*sizeof(double));
	gl=(double**)malloc((Height+1)*sizeof(double*));
 	if ((tmp==NULL)||(gl==NULL))
	{
        fprintf(stderr,"imgD_resize: out of memory.\n"); exit(222);
	};	
	tmp--; for (i=1;i<=Height;i++, tmp+=Width) gl[i]=tmp;
	fft=NULL;
	
	if ((scaleX==1)&&(scaleY==1))
	{
		memcpy(gl[1]+1,im->gl[1]+1,Size*sizeof(double));
		if (im->fft==NULL) return;
		// copier aussi la FFT si elle a t calcule.
		tmp=(double*)malloc(Size*sizeof(double));
		fft=(double**)malloc((Height+1)*sizeof(double*));
		speq=(double*)malloc((2*Height+1)*sizeof(double));
     	if ((tmp==NULL)||(fft==NULL)||(speq==NULL))
	    {
            fprintf(stderr,"imgD_resize: out of memory.\n"); exit(223);
        };
		tmp--; for (i=1;i<=Height;i++, tmp+=Width) fft[i]=tmp;
		memcpy(fft[1]+1,im->fft[1]+1,Size*sizeof(double));
		memcpy(speq,im->speq,(2*Height+1)*sizeof(double));
		return;
	};
	if (scaleX<1)
	{
		// scaleY must also be <=1
		incX=(DWORD)(65536*scaleX);
		incY=(DWORD)(65536*scaleY);
		memset(gl,0,Size);
		indexDstartline=gl[1]+1;
		indexS=im->gl[1]+1;
		counterY=0;
		surfaceStartline=surfaceStartimage=(double*)malloc(Size*sizeof(double));
		if (surfaceStartline==NULL)
		{
            fprintf(stderr,"imgD_resize: out of memory.\n"); exit(223);
        };
		memset(surfaceStartline,0,Size*sizeof(double));
		for (j=0; j<im->Height; j++)
		{
			counterX=0;
			indexD=indexDstartline;
			surface=surfaceStartline;
			for (i=0; i<im->Width; i++)
			{
				a=(DWORD)(counterX>>16);
				indexD[a]++;
				surface[a]+=*(indexS++);
				counterX+=incX;
			};
			counterY+=incY;
			if (counterY>65535)
			{
				indexDstartline+=Width;
				surfaceStartline+=Width;
				counterY&=65535;
			};
		};
		indexD=*gl; indexS=indexD+Size;
		surface=surfaceStartimage;
		while (indexD!=indexS)
		{
			b=((*indexD!=0)?((*surface)/(*indexD)):0);
			if (b==255) *indexD=254; else *indexD=b;
			indexD++;
			surface++;
		};
		free(surfaceStartimage);
	} else
	{
		indexSstartline=im->gl[1]+1;
		indexD=gl[1]+1;
		incX=(DWORD)(65536/scaleX);
		incY=(DWORD)(65536/scaleY);
		counterY=0;
		for (j=0; j<Height; j++)
		{			
			counterX=0;
			indexS=indexSstartline;
			for (i=0; i<Width; i++)
			{
				(*(indexD++))=indexS[(int)(counterX>>16)];
				counterX+=incX;
			};
			counterY+=incY;
			if (counterY>65535) 
			{
				indexSstartline+=im->Width;
				counterY&=65535;
			};
		};
	};
};

imageD::imageD(WORD _Width,WORD _Height,double c)
{
	double *tmp,*tmp2;
	DWORD i;
	Height=_Height;
	Width=_Width;
	Size=Height*Width;
	tmp2=tmp=(double*)malloc(Size*sizeof(double));
    gl=(double**)malloc((Height+1)*sizeof(double*));
 	if ((tmp2==NULL)||(gl==NULL))
	{
        fprintf(stderr,"imgD_create: out of memory.\n"); exit(222);
	};
	for (i=0;i<Size;i++) *(tmp2++)=c;
	tmp--; for (i=1;i<=Height;i++, tmp+=Width) gl[i]=tmp;
	fft=NULL;
};

imageD::~imageD()
{
	free(gl[1]+1);
	free(gl);
	if (fft!=NULL)
	{
		free(fft[1]+1);
		free(fft);
		free(speq);
	};
};

void imageD::find_max(WORD *x,WORD *y)
{
	double *glf=gl[1]+1,*glend=glf+Size,*gli=glf+1,a;
	DWORD l;
		
	while (glend!=gli) 
		if (*(gli++)>a) { glf=gli-1; a=*glf; };
	l=(DWORD)(glf-(gl[1]+1));
	*x=(WORD)(l%Width)+1;
	*y=(WORD)(l/Width)+1;
};

/*
void imageD::normaliseFFT()
{
	DWORD j;
	double *sp1;
	double fac=2.0/(Height*Width); // Factor needed to get normalized inverse.
	sp1 = fft[1]+1;
	for (j=0;j<(DWORD)(Height*Width/2);j++) 
	{
		sp1[0] *= fac;
		sp1[1] *= fac;
		sp1 += 2;		
	}
	sp1 = speq+1;	
	for (j=0;j<(DWORD)Height;j++) 
	{
		sp1[0] *= fac;
		sp1[1] *= fac;
		sp1 += 2;		
	}
}

void imageD::cutHighFreq(int n)
{
	int i,j;
	normaliseFFT();	
	for (j=Height-n+1; j<=Height; j++)
		for (i=1; i<=Width; i++)
			fft[j][i]=0; 

	for (j=Width/2-n+2; j<=Width/2; j++)
		for (i=1; i<=Height; i++)
		{
			fft[i][(j<<1)]=0;
			fft[i][(j<<1)-1]=0;
		};
	for (i=1; i<=2*Height; i++) speq[i]=0; 

	memcpy(gl[1]+1,fft[1]+1,Size*sizeof(double));
	rlft3(&gl-1,&speq-1,1,Height,Width,-1); // Inverse FFT the product of the two FFTs.
};
*/