#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
#include "simpleGA.h"

#define SQR(a) ((a)*(a))
#define MAX(a,b) ((a)>(b)?(a):(b))
#define MIN(a,b) ((a)<(b)?(a):(b))
static const double PI=2*asin(1);

inline double MYRAND()
{
    double d=((double)(rand()-1))/RAND_MAX;
    return MAX(0.0,d);
}

class Population;

class Individu
{
  public:
    double *x;
    double quality;
    char evaluated;
    Population *p;

    Individu(Population *_p);
    Individu(double pa, Individu *a, double pb, Individu *b);
    Individu(Individu *a, Individu *b);
    Individu(Individu *a);
    ~Individu() {free(x);}
    void mutate();
    void eval();
    void Rinit();
};

class Population
{
  public:
    int nbrEval;
    double *bl,*bu;
    Individu **p;
    void (__cdecl *f)(int,double *,double *,void *); // quality function
    void *extraparams;
    int n, sdim; // n: number of Individual
                 // sdim: search space dimension
    Population(int _n, int _sdim, double *bl, double *bu, 
        void (__cdecl *_f)(int,double *,double *,void *), void *);
    ~Population();
    void eval();
    void sort();
    void unsort();
    void selection1();
    void selection2();
    void crossover();
    void mutation();
    void evolue();
    void bestIndividuFirst();
    int best(int s,int e);
    int worse(int s,int e);
};

Individu::Individu(Population *_p): p(_p), evaluated(false)
{ 
    x=(double*)malloc(p->sdim*sizeof(double));
}

void Individu::Rinit()
{
    int i;
    for (i=0; i<p->sdim; i++) x[i]=p->bl[i]+MYRAND()*(p->bu[i]-p->bl[i]);
};

void Individu::eval() 
{
    if (!evaluated)
    {
        (*(p->f))(p->sdim,x,&quality,p->extraparams);
        evaluated=true;
        p->nbrEval++;
    };
};

Individu::Individu(double pa, Individu *a, double pb, Individu *b) : p(a->p), evaluated(false)
{
    int k;
    x=(double*)malloc(p->sdim*sizeof(double));
    for (k=0; k<p->sdim; k++) 
        x[k]=MAX(MIN(pa*a->x[k]+pb*b->x[k],p->bu[k]),p->bl[k]);
}

Individu::Individu(Individu *a) : p(a->p), quality(a->quality), evaluated(a->evaluated)
{
    x=(double*)malloc(p->sdim*sizeof(double));
    memcpy(x,a->x,p->sdim*sizeof(double));
};

void Individu::mutate()
{
    // normal cauchy distribution : f(x)=1/(PI*(1+x^2))
    // inverse cumulative distribution function is: x=-cotg(PI*p)
    int i=(int)(MYRAND()*p->sdim);
    double delta=(p->bu[i]-p->bl[i])*PERTURBATION;
    x[i]=MAX(MIN(x[i]-delta/tan(PI*MYRAND()),p->bu[i]),p->bl[i]);
}

Population::Population(int _n, int _sdim, double *_bl, double *_bu, 
                       void (__cdecl *_f)(int,double *,double *,void *), void *e): 
    n(_n) , sdim(_sdim), f(_f), bl(_bl), bu(_bu), extraparams(e), nbrEval(0)
{ 
    int i;
    srand( (unsigned)time( NULL ) );
    p=(Individu**)malloc(n*sizeof(Individu*));
    for (i=0; i<n; i++)
    {
        p[i]= new Individu(this);
        p[i]->Rinit();
    }
};

Population::~Population()
{
    int i;
    for (i=0; i<n; i++) delete p[i];
    free(p);
}

void Population::bestIndividuFirst()
{
    int max=best(0,n);
    Individu *t=p[0]; 
    p[0]=p[max]; 
    p[max]=t;
}

void Population::eval()
{
    int i;
    for (i=0; i<n; i++) p[i]->eval();
}

void Population::sort()
{
    // tri par insertion
    int i,j;
    Individu *toSort;
    for (j=1; j<n; j++)
    {
        i=j-1; toSort=p[j];
        while ((i>=0)&&(toSort->quality>p[i]->quality)) i--;
        i++;
        if (i<j)
        {
            memmove(p+i+1,p+i,(j-i)*sizeof(Individu*));
            p[i]=toSort;
        };
    }
}

void Population::unsort()
{
    int i,j,nn=n,r;
    Individu **newp=(Individu **)malloc(n*sizeof(Individu *));

    memset(newp,0,n*sizeof(Individu *));
    for (i=0; i<n; i++)
    {
        r=(int)(MYRAND()*nn);
        j=-1;
        while (r>=0) 
        {
            j++;
            if (newp[j]==NULL) r--;
        };
        newp[j]=p[i];
        nn--;
    }
    free(p);
    p=newp;
}

void Population::selection1()
{
    // basic selection 
    // probability to choose Individual i is proportional to it's quality
    Individu **newp=(Individu **)malloc(n*sizeof(Individu *));
    double sum=0,sum2,r;
    int i,j;

    bestIndividuFirst();
    for (i=0; i<n; i++) sum+=p[i]->quality;
    newp[0]=p[0];
    for (i=1; i<n; i++)
    {
        r=MYRAND()*sum;
        sum2=0; j=0;
        while (r>=sum2) 
        {
            sum2+=p[j]->quality;
            j++;
        }
        newp[i]=new Individu(p[j-1]);
    }
    for (i=1; i<n; i++) delete p[i]; // i=1 because we have made newp[0]=p[0];
    free(p);
    p=newp;
}

void Population::selection2()
{
    Individu **newp=(Individu **)malloc(n*sizeof(Individu *));

    // selection scheme from:
    // M.D. Vodes. "Simple genetic algorithm with linear scaling" in
    // evolutionary computation, 2(4): 347-368, 1995

    // probability to choose Individual i is proportional to it's rank
    // g= selective pressure (1<=g<=2)

    double sum2,r,g=1.5,sum=n*g-(2*g-2)*(n*(n+1)*.5-1)/(n-1);
    int i,j;

    sort();
    newp[0]=p[0];
    for (i=1; i<n; i++)
    {
        r=MYRAND()*sum;
        sum2=0; j=0;
        while (r>=sum2) 
        {
            sum2+=g-(2*g-2)*(j-1)/(n-1);
            j++;
        }
        newp[i]=new Individu(p[j-1]);
    }
    for (i=1; i<n; i++) delete p[i]; // i=1 because we have made newp[0]=p[0];
    free(p);
    p=newp;
}

int Population::best(int s,int e)
{
    int j,max=s;
    for (j=s+1; j<e; j++)
        if (p[j]->quality>p[max]->quality) max=j;
    return max;
}

int Population::worse(int s,int e)
{
    int j,min=s;
    for (j=s+1; j<e; j++)
        if (p[j]->quality<p[min]->quality) min=j;
    return min;
}

void Population::crossover()
{
    Individu *max,*min,*centroid,*reflect,*expand,*contract,*discrete,*average;
    int i=0,j,k,w; // is= index source        id= index destination
    double r;
    double Pd=PD,Pc=PC,Ps=PS;
    unsort();
    while (i<n)
    {
        if (i+sdim+1>n) Ps=0;

        w=worse(i,MIN(i+sdim+1,n));
        r=MYRAND();
        if (r<=Pd)
        {
            // discrete crossover
            discrete=new Individu(this);
            for (k=0; k<sdim; k++) 
                discrete->x[k]=p[i+(int)(MIN(sdim+1,n-i)*MYRAND())]->x[k];
            delete p[w];
            p[w]=discrete;
        }
        if ((Pd<r)&&(r<=Pd+Pc))
        {
            // average crossover
            average=new Individu(this);
            memset(average->x, 0, sdim*sizeof(double));
            for (j=i; j<MIN(n,i+sdim+1); j++)
                for (k=0; k<sdim; k++) average->x[k]+=p[j]->x[k];
            delete p[w];
            p[w]=average;
        }

        if ((Pd+Pc<r)&&(r<=Pd+Pc+Ps))
        {
            // simplex crossover

            // 1) find min and max
            max=p[best(i,i+sdim+1)]; 
            min=p[w];

            // 2) calculate centroid
            centroid=new Individu(this);
            memset(centroid->x, 0, sdim*sizeof(double));
            for (j=i; j<i+sdim+1; j++)
                if (j!=w) for (k=0; k<sdim; k++) centroid->x[k]+=p[j]->x[k];
            for (k=0; k<sdim; k++) centroid->x[k]/=sdim;

            reflect=new Individu(2, centroid, -1, min);
            reflect->eval();
            if (reflect->quality>max->quality)
            {
                expand=new Individu(2, reflect, -1, centroid);
                expand->eval();
                if (expand->quality>reflect->quality) 
                {
                    delete min;
                    p[w]=expand;
                    delete centroid;
                    delete reflect;
                } else
                {
                    delete min;
                    p[w]=reflect;
                    delete centroid;
                    delete expand;
                }
            } else
            {
                if (reflect->quality>min->quality)
                {
                    delete min;
                    p[w]=reflect;
                    delete centroid;
                } else
                {
                    contract=new Individu(.5, min, .5, centroid);
                    contract->eval();
                    if (contract->quality>min->quality)
                    {
                        delete min;
                        p[w]=contract;
                        delete centroid;
                    } else
                    {
                        p[w]=new Individu(.5, min, .5, max);
                        delete min;
                        delete centroid;
                        delete contract;
                    }
                }
            }
        }
        i+=sdim+1;
    }
}

void Population::mutation()
{
    int i;
    // normal cauchy distribution : f(x)=1/(PI*(1+x^2))
    // inverse cumulative distribution function is: x=-cotg(PI*p)
    bestIndividuFirst();
    for (i=1; i<n; i++)
        if (MYRAND()<MR) p[i]->mutate();
}

void Population::evolue()
{
     eval();
     selection2();
     crossover();
     mutation();
     eval();
};

void strategy1(Population *pop, int maxiter, int verbose)
{
//   1st strategy for stop:
    int i;
    for (i=0; i<maxiter; i++) 
    {
        pop->evolue();
        if (verbose) printf("best quality at generation %i: Q=%f\n",
                             i+1,pop->p[0]->quality);
    };
    pop->bestIndividuFirst();
}

void strategy2(Population *pop, int verbose)
{
//   2nd strategy for stop:
    int i=1;
    do
    {
        pop->evolue();
        pop->bestIndividuFirst();
        if (verbose) printf("best quality at generation %i: Q=%f\n",
                             i,pop->p[0]->quality);
        i++;
    } while (pop->p[0]->quality<-0.05);
}

void simpleGA(int n, double *x, double *bl, double *bu, int maxiter, int verbose,
				void obj(int,double *,double *,void *), void *extraparams)
{
    Population *pop=new Population(100,n,bl,bu,obj,extraparams);
    if (verbose) printf("GA method for global optimization (maximization)\n\n");

//    strategy1(pop,maxiter,verbose);
    strategy2(pop,verbose);

    memcpy(x,pop->p[0]->x,n*sizeof(double));
    if (verbose)
        printf("\nnumber of evaluation of the objective function= %i\n\n",pop->nbrEval);
    delete pop;
}
