function m = mapping(varargin)
%MAPPING Constructor for the mapping abstract class
%    
%    M=MAPPING creates a new general mapping skeleton
%    
%    M=MAPPING(NAME,N_IN,N_OUT) creates a new general mapping
%    with N_IN inputs and N_OUT outputs. The name NAME is
%    associated to it.
%    
%    The class defines the following methods:
%    
%      CHECK: checks the consistency of all the field of a
%          mapping object
%      DENORMALISE: normalises the model M
%      DISPLAY: Display an object of class mapping
%      ERROR: Computes the squared error of the mapping
%      EVAL: computes the value of the mapping for some
%          input
%      GET: gets the value of the attribute of an object
%      IDENTIFY: Identify the mapping from data (abstact
%          method)
%      JACOB_INPUTS: computes the jacobian of the model
%      JACOB_PARAMS: computes the jacobian of the model
%      LEV_MARQ: Performs a Levenbergh Marquardt
%          optimisation of the mapping
%      NORMALISE: normalises the model M
%      SET: Set object properties
%    
%    See also: 
%    
%    

%MAN_PAGE_BEGIN
%
%   @purpose	Constructor for the mapping abstract class
%
%   @synopsis	m=mapping
%   @description  creates a new general mapping skeleton
%
%   @synopsis	m=mapping(name,n_in,n_out)
%   @description  creates a new general mapping with <CODE>n_in</CODE> inputs
%	and <CODE>n_out</CODE> outputs. The name <CODE>name</CODE> is associated to it.
%	
%
%   @see 
%
%MAN_PAGE_END

error(checkargs(varargin,[0 3],'char','numeric','numeric'));

m=[];
if nargin == 0,
	m.name = 'untitled mapping';
	m.n_in=0;                       %number of inputs 
	m.n_out=0;                      %number of outputs
	m.limits=[];
	m.userData=[];
	m.opt=[];
	m.date = clock;                 %creation date
	m.optimparams = [];
	m = class(m,'mapping');
elseif nargin == 1,
	m.name = varargin{1};
	m.n_in=1;                       %number of inputs 
	m.n_out=1;                      %number of outputs
	m.limits=[-inf -inf;inf inf];
	m.userData=[];
	m.opt=[];
	m.date = clock;                 %creation date
	m.optimparams = [];
	m=class(m,'mapping');
elseif nargin == 2,
	m.name = varargin{1};
	m.n_in = varargin{2};           %number of inputs 
	m.n_out=1;                      %number of outputs
	m.limits=[[-inf;inf] * ones(1,varargin{2}) [-inf; inf]];
	m.userData=[];
	m.opt=[];
	m.date = clock;                 %creation date
	m.optimparams = [];
	m=class(m,'mapping');
	
	if m.n_in < 0,
		error('The number of inputs must be positive');
	end
	if m.n_out < 0,
		error('The number of outputs must be positive');
	end
	
elseif nargin == 3,
	m.name = varargin{1};
	m.n_in = varargin{2};           %number of inputs 
	m.n_out = varargin{3};                      %number of outputs
	m.limits=[[-inf;inf] * ones(1,varargin{2}) [-inf; inf] * ones(1,varargin{3})];
	m.userData=[];
	m.opt=[];
	m.date = clock;                 %creation date
	m.optimparams = [];
	m=class(m,'mapping');
	
	
	if m.n_in < 0,
		error('The number of inputs must be positive');
	end
	if m.n_out < 0,
		error('The number of outputs must be positive');
	end
end
	
	
