function m = set(m,varargin);
%SET Set object properties
%    
%    SET(M,'ATTRNAME1',ATTRVALUE1,'ATTRNAME2',ATTRVALUE2,...)
%    sets the value of the attribute ATTRNAME1, ATTRNAME2, ...
%    associated to the object M.
%    
%    M_OUT =
%    SET(M,'ATTRNAME1',ATTRVALUE1,'ATTRNAME2',ATTRVALUE2,...)
%    sets the value of the attribute ATTRNAME1, ATTRNAME2, ...
%    associated to the object M and return the modified model
%    inside m_out. The original model is not modified. The
%    following ATTRNAME are recognised:
%    
%      NAME: sets the name of the object
%      N_IN: sets the number of inputs of the model. This
%          sets the saturation level of the inputs to  -Inf,
%          Inf.
%      N_OUT: sets the number of outputs of the model
%      USERDATA: sets the 'userData' field
%      OPT: sets the 'opt' field
%      DATE: sets the date of creation of the object
%      LIMITS: sets the limits of the domain of definition
%          of the mapping. The value must be a 2*N_IN matrix
%          where the upper and lever saturation level are
%          defined.
%      OPTIMPARAMS: sets the indices of the parameters to be
%          optimised
%    
%    See also: GET
%    
%    

%MAN_PAGE_BEGIN
%
%		@purpose	Set object properties.
%
%		@synopsis		set(m,'AttrName1',AttrValue1,'AttrName2',AttrValue2,...)
%		@description	sets the value of the attribute <code>AttrName1</code>, <code>AttrName2</code>, ...
%		associated to the object <code>m</code>.
%
%		@synopsis		m_out = set(m,'AttrName1',AttrValue1,'AttrName2',AttrValue2,...)
%		@description	sets the value of the attribute <code>AttrName1</code>, <code>AttrName2</code>, ...
%		associated to the object <code>m</code> and return the modified model inside m_out. The original
%		model is not modified.
%
%		The following <CODE>AttrName</CODE> are recognised:
%
%<dl>
%<dt>			name		 <dd>		sets the name of the object
%<dt>			n_in		 <dd>		sets the number of inputs of the model. This
%													sets the saturation level of the inputs to 
%													-Inf, Inf.
%<dt>			n_out		 <dd>		sets the number of outputs of the model
%<dt>			userData <dd>		sets the 'userData' field
%<dt>			opt			 <dd>		sets the 'opt' field
%<dt>			date		 <dd>		sets the date of creation of the object
%<dt>	 limits	 <dd>				sets the limits of the domain of definition
%								of the mapping. The value must be a <code>2*n_in</code> matrix where
%													the upper and lever saturation level are
%													defined.
%<dt>	 optimparams	 <dd>		sets the indices of the parameters to be optimised	
%</dl>
%
%		@see get
%
%MAN_PAGE_END		


% Check that we have the right number of inputs
ni = nargin;
if (ni<3)|(ni/2 == floor(ni/2)),
  error('Wrong number of inputs'); %
end

if ~isa(m,'mapping'),
  % Call built-in SET. Handles calls like set(gcf,'user',taksug)	
  builtin('set',m,varargin{:});
  return
end

% For each couple of inputs
for i=1:(ni - 1)/2,
  name = varargin{i*2-1};
  value = varargin{i*2};
  
  if isstr(name),
    switch lower(name),
      
     case 'name',
      error(checkargs({value},[1],'char'));
      m.name = value;
     case 'n_in',
      error(checkargs({value},[1],'numeric'));
      if (value<1)
	error('number of inputs must be bigger than 0');
      else 
	m.n_in = value;
	m.limits=[[-inf;inf] * ones(1,m.n_in) ...
		  [-inf; inf] * ones(1,m.n_out)]; 
      end
      
     case 'n_out',
      error(checkargs({value},[1],'numeric'));
      if (value<1)
	error('number of inputs must be bigger than 0');
      else 
	m.n_out = value;
	m.limits=[[-inf;inf] * ones(1,m.n_in) ...
		  [-inf; inf] * ones(1,m.n_out)]; 
      end
      
     case 'userdata',
      m.userData = value;
      
     case 'optimparams',
      m.optimparams = value;
      
     case 'opt',
      m.opt = value;
      
     case 'date',
      if (isa(value,'double') & size(value) == [1 6])
	m.date = value;
      else 
	error('invalid input argument for field ''date'''); 
      end
      
     case 'limits',
      if isfield(m.opt,'original_limits'),
	error('Cannot chanage the limits of a normalized function');
      end
      if size(value) == [2 m.n_in + m.n_out],
	for j=1:m.n_in+m.n_out,
	  if value(1,j)>value(2,j),
	    error('inconsistent values for the field ''limits''');
	  end
	end	 
	m.limits=value;
      else 
	error('invalid input argument for field ''limits''');
      end	
     otherwise
      error([''''name ''' is not a property of this object']);
    end
  else
    error('Arguments mismatch');
  end
end

if nargout == 0,
  assignin('caller',inputname(1),m)
end

