function m = set(m,varargin);
%SET set object properties
%    
%    General purpose function for  setting the values of the
%    fields of a taksug object. For setting	the rules, use
%    the ADD_RULES and REM_RULES function. This method extends
%    the 'set' method defined in the mpping class. 
%    
%    M=SET(M,LABEL1,VALUE1,LABEL2,VALUE2,...) sets the value
%    VALUE of the attribute LABEL associated to the object M.
%    The following codes are recognised:
%    
%      NAME: sets the name of the object
%      N_IN: sets the number of inputs of the model. This
%          sets the saturation level of the inputs to  -Inf,
%          Inf.
%      N_OUT: sets the number of outputs of the model
%      USERDATA: sets the 'userData' field
%      OPT: sets the 'opt' field
%      DATE: sets the date of creation of the object
%      LIMITS: sets the limits of the mapping
%      OPTIMPARAMS: sets the indices of the parameters to be
%          optimised
%      MODEL_CODE: sets the type of rules associated to the
%          taksug model
%      M: sets the fuzziness associated to the taksug model
%      CENTERS: sets the positions of the centers of the
%          rules  associated to the taksug model
%      CENTER: sets the positions of one center of the rules
%           associated to the taksug model. VALUE is of the
%          form {number value}.
%      IVARIANCES: sets the invariances of the rules 
%          associated to the taksug model
%      IVARIANCE: sets the positions of one ivariance of the
%          rules  associated to the taksug model. VALUE is
%          of the form {number value}.
%      LINEARS: sets the linears of associated to the taksug
%          model
%      LINEAR: sets the positions of one linear of the rules
%           associated to the taksug model. VALUE is of the
%          form {number value}.
%      RLS: sets the fuzzy sets associated to the taksug
%          model
%      RL: sets one fuzzy set associated to the taksug model
%      MFS: sets the definition of the rules of the taksug
%          model
%      MF: sets the definition of one rule of the taksug
%          model
%      PARAMS: sets every parameters to be optimised
%          providing a vector
%      OPTIMSELECT: basically achieves the same result as
%          OPTIMPARAMS but lets the user perform a grouped
%          selection of the parameters (all the consequents
%          parameters, all the centers,...), VALUE is 4 bits
%          binary number dcba. a is set for selecting the
%          centers, b for the ivariances, c for the
%          consequents except the offsets, d for the offsets
%    
%    See also: GET, ADD_RULES, REM_RULES
%    
%    

%MAN_PAGE_BEGIN
%
%		@purpose	set object properties. General purpose function for 
%		setting the values of the fields of a taksug object. For setting 
%		the rules, use the <code>add_rules</code> and <CODE>rem_rules</CODE> function. This method
%		extends the 'set' method defined in the mpping class. 
%
%		@synopsis		m=set(m,label1,value1,label2,value2,...)
%		@description	sets the value <code>value</code> of the attribute <code>label</code>
%		associated to the object <code>m</code>. The following codes are recognised:
%<dl>
%<dt>			name		 <dd>		sets the name of the object
%<dt>			n_in		 <dd>		sets the number of inputs of the model. This
%													sets the saturation level of the inputs to 
%													-Inf, Inf.
%<dt>			n_out		 <dd>		sets the number of outputs of the model
%<dt>			userData <dd>		sets the 'userData' field
%<dt>			opt			 <dd>		sets the 'opt' field
%<dt>			date		 <dd>		sets the date of creation of the object
%<dt>	 limits	 <dd>				sets the limits of the mapping
%<dt>	 optimparams	 <dd>		sets the indices of the parameters to be optimised
%<dt> model_code   <dd>   sets the type of rules associated to the
%                         taksug model
%<dt> m            <dd>   sets the fuzziness associated to the
%                         taksug model
%<dt> centers      <dd>   sets the positions of the centers of the
%                         rules  associated to the taksug model
%<dt> center      <dd>   sets the positions of one center of the
%                         rules  associated to the taksug model. <CODE>value</CODE>
%							is of the form {number value}.
%<dt> ivariances  <dd>    sets the invariances of the
%                         rules  associated to the taksug model
%<dt> ivariance      <dd>   sets the positions of one ivariance of the
%                         rules  associated to the taksug model. <CODE>value</CODE>
%							is of the form {number value}.
%<dt> linears  <dd>       sets the linears of associated to the taksug
%                         model
%<dt> linear      <dd>   sets the positions of one linear of the
%                         rules  associated to the taksug model. <CODE>value</CODE>
%							is of the form {number value}.
%<dt> rls  <dd>           sets the fuzzy sets associated to the taksug
%                         model
%<dt> rl  <dd>           sets one fuzzy set associated to the taksug
%                         model
%<dt> mfs  <dd>       sets the definition of the rules of the taksug model
%<dt> mf  <dd>       sets the definition of one rule of the taksug model
%<dt> params  <dd>       sets every parameters to be optimised providing a vector
%<dt> optimselect  <dd>       basically achieves the same result as <CODE>optimparams</CODE>
%							but lets the user perform a grouped selection of the parameters
%							(all the consequents parameters, all the centers,...), <CODE>value</CODE>
%							is 4 bits binary number dcba. a is set for selecting the centers, b for the
%							ivariances, c for the consequents except the offsets, d for the offsets
%</dl>
%	 
%
%		@see get, add_rules, rem_rules
%
%MAN_PAGE_END		


ni = nargin;
if (ni<3)|(ni/2 == floor(ni/2)),
	error('Wrong number of inputs');
end

if ~isa(m,'taksug'),
	
	% Call built-in SET. Handles calls like set(gcf,'user',linears) 
	builtin('set',m,varargin{:});
	return

end

% For each couple of inputs
for i=1:(ni - 1)/2,
	name = varargin{i*2-1};
	value = varargin{i*2};
	
	switch name,
			
	case 'centers',
		if iscompat(value,zeros(get(m,'n_in'), m.n_rules)),
			m.centers = value;
		else
			error(['Size of the new value (' num2str(size(value)) ') for centers is not adequate.' 13 'It should be ' num2str([get(m,'n_in') m.n_rules])]);
		end
		
	case 'center',
	error(checkval(value,'cell', 'center property'));
	error(checkargs(value, 2, 'numeric', 'numeric'));
	m.centers(:,value{1}) = value{2}(:);	

	case 'ivariance',
	error(checkval(value,'cell', 'ivariance property'));
	error(checkargs(value, 2, 'numeric', 'numeric'));
	m.ivariances(:,:,value{1}) = value{2};	

	case 'linear',
	error(checkval(value,'cell', 'linear property'));
	error(checkargs(value, 2, 'numeric', 'numeric'));
	m.linears(:,:,value{1}) = value{2};	

	case 'rl',
	error(checkval(value,'cell', 'linear property'));
	error(checkargs(value, 2, 'numeric', 'numeric'));
	m.rls(value{1},:) = value{2};	

	case 'mf',
	error(checkval(value,'cell', 'linear property'));
	error(checkargs(value, 3, 'numeric', 'numeric', 'numeric'));
	m.mfs{value{1}}(value{2},2:end) = value{3};	

	case 'mftype',
	error(checkval(value,'cell', 'linear property'));
	error(checkargs(value, 3, 'numeric', 'numeric', 'char'));
	
	switch lower(value{3}),
		case 'constant',
		code = 0;
		
		case 'trapezoidal',
		code = 1;
		
		case 'gaussian',
		code = 2;
		
		case 's-shaped',
		code = 3;
		
		otherwise
		error('Unrecognized rule type');
	end
	
	m.mfs{value{1}}(value{2},1) = code;	

	case 'ivariances',
		if iscompat(value,zeros(get(m,'n_in'), get(m,'n_in'), m.n_rules)),
			m.ivariances = value;
		else
			error(['Size of the new value (' num2str(size(value)) ') for ivariances is not adequate.' 13 'It should be ' num2str([get(m,'n_in') get(m,'n_in') m.n_rules])]);
		end

	case 'rls',
		if iscompat(value,zeros(m.n_rules, get(m,'n_in'))),
			m.rls = value;
		else
			error(['Size of the new value (' num2str(size(value)) ') for rls is not adequate.' 13 'It should be ' num2str([m.n_rules get(m,'n_in')])]);
		end

	case 'mfs',
		m.mfs = value;
		
	case 'linears',
		if iscompat(value,zeros(get(m,'n_out'), get(m,'n_in')+1, m.n_rules)),
			m.linears = value;
		else
			error(['Size of the new value (' num2str(size(value)) ') for linears is not adequate.' 13 'It should be ' num2str([get(m,'n_in') get(m,'n_out')+1 m.n_rules])]);
		end
		
	case 'm',
		if iscompat(value , 0),
			m.m = value;
		else
			error(['Size of the new value (' num2str(size(value)) ') for m should be 1']);
		end
		
	case 'model_code',
		if isequal(size(value),[1 3]),
			m.model_code = value;
		else
			error(['Size of the new value (' num2str(size(value)) ') for model_code is not adequate']);
		end
	
	case 'params'
		ni = get(m,'n_in');
		no = get(m,'n_out');
		nr = get(m,'n_rules');
		temp = [m.centers(:)' m.ivariances(:)' m.linears(:)'];
		temp(1,get(m,'optimparams')) = value;
		m.centers(:) = temp(1:ni*nr);
		m.ivariances(:) = temp(1+ni*nr:(ni+1)*ni*nr);
		m.linears(:) = temp(1+(ni+1)*ni*nr:(ni+1)*(ni+no)*nr);
		
	case 'optimselect',
		if iscompat(value, 0),
		  set(m,'optimparams',sub_params(m,value));
		else
		  error(['optimparams should be scalar']);
		end
	otherwise
		m.mapping = set(m.mapping, name, value);
	
	end
	
end

if nargout == 0,
	assignin('caller',inputname(1),m)
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function list=sub_params(m,select)
  

out = zeros(m.n_rules * (get(m,'n_in')+get(m,'n_in')^2+(get(m,'n_in')+1)* ...
			 get(m,'n_out')),1);
list=[1:size(out,1)];

beg_mark = 1;
end_mark = get(m,'n_in') * m.n_rules;

if bitand(1,select),
	out(beg_mark:end_mark) = 1;
end

beg_mark = end_mark + 1;
end_mark = end_mark + get(m,'n_in') ^ 2 * m.n_rules;


if bitand(2,select), 
	out(beg_mark:end_mark) = 1;
end

beg_mark = end_mark + 1;
end_mark = end_mark + (1+ get(m,'n_in')) * get(m,'n_out') * m.n_rules;

if bitand(4,select),
	out(beg_mark:end_mark) = 1;
end

end_mark = beg_mark - 1;

for i=1:m.n_rules,
	beg_mark = end_mark + get(m,'n_out') * get(m,'n_in') + 1;
	end_mark = end_mark + get(m,'n_out') * (1 + get(m,'n_in'));		
	if bitand(8,select),
		out(beg_mark:end_mark) = 1;
	else
		out(beg_mark:end_mark) = 0;
	end
end

list=list(1,find(out>0)');

