function d = set(d,varargin)
%SET Set object properties
%    
%    SET(D,'ATTRNAME1',ATTRVALUE1,'ATTRNAME2',ATTRVALUE2,...)
%    sets the value of the attribute ATTRNAME1, ATTRNAME2, ...
%    associated to the object D.
%    
%    M_OUT =
%    SET(D,'ATTRNAME1',ATTRVALUE1,'ATTRNAME2',ATTRVALUE2,...)
%    sets the value of the attribute ATTRNAME1, ATTRNAME2, ...
%    associated to the object D and return the modified model
%    inside m_out. The original model is not modified. The
%    following ATTRNAME are recognised:
%    
%      CONTDATA: the continuous data
%      SYMBDATA: the symbolic data
%      CLASSES: the classes
%      LABELS: the labesl of the data points
%      VARTYPES: the types of the variables
%    
%    See also: GET
%    
%    

%MAN_PAGE_BEGIN
%
%		@purpose	Set object properties.
%
%		@synopsis		set(d,'AttrName1',AttrValue1,'AttrName2',AttrValue2,...)
%		@description	sets the value of the attribute <code>AttrName1</code>, <code>AttrName2</code>, ...
%		associated to the object <code>d</code>.
%
%		@synopsis		m_out = set(d,'AttrName1',AttrValue1,'AttrName2',AttrValue2,...)
%		@description	sets the value of the attribute <code>AttrName1</code>, <code>AttrName2</code>, ...
%		associated to the object <code>d</code> and return the modified model inside m_out. The original
%		model is not modified.
%
%		The following <CODE>AttrName</CODE> are recognised:
%
%  	<dl>
%	<dt> data <dd> the data (but not the classes)
%	<dt> contdata <dd> the continuous data
%	<dt> symbdata <dd> the symbolic data
%	<dt> classes <dd> the classes (either as cell array either as real data)
%	<dt> labels <dd> the labes of the data points
%	<dt> vartypes <dd> the types of the variables
%	</dl>
%
%		@see get
%
%MAN_PAGE_END		


% Check that we have the right number of inputs
ni = nargin;
if (ni<3)|(ni/2 == floor(ni/2)),
   error('Wrong number of inputs'); %
end

if ~isa(d,'dataset'),
   % Call built-in SET. Handles calls like set(gcf,'user',system)	
   builtin('set',d,varargin{:});
   return
end

% For each couple of inputs
for i=1:(ni - 1)/2,
   name = varargin{i*2-1};
   value = varargin{i*2};
   
   if isstr(name),
      switch lower(name),
         
		 %
         %	contdata property
         %	-------------		
      case 'data',
         error(checkval(value,'real','data ', ...
            {size(d,1) sum(~strcmp(d.vartypes,'class')) 0}));
         d.data(:,find(~strcmp(d.vartypes,'class'))) = value;
         
         %
         %	contdata property
         %	-------------		
      case 'contdata',
         error(checkval(value,'real','continuous data ', ...
            {size(d,1) sum(strcmp(d.vartypes,'continuous')) 0}));
         d.data(:,find(strcmp(d.vartypes,'continuous'))) = value;
         
         %
         %	symbdata property
         %	-------------		
      case 'symbdata',
         error(checkval(value,'real','symbolic data ', ...
            {size(d,1) sum(strcmp(d.vartypes,'symbolic')) 0}));
         d.data(:,find(strcmp(d.vartypes,'symbolic'))) = value;
         
         %
         %	classes property
         %	-------------		
      case 'classes',
         if isempty(checkval(value,'real','classes ',{size(d,1) 1 0}))
            col = find(strcmp(d.vartypes,'class'));
            
            if isempty(col),
            	col = size(d,2) + 1;
            	d.vartypes{col} = 'class';
				d.variables{col} = 'class';
				d.symbols{col} = {};
				if col == 1,
					d.labels = (1:size(value,1))'
				end
            end
            
            maxval = max(value);
            
            if maxval>length(d.symbols{col}),
            	ds = cell(1,maxval);
            	for i=1:maxval,
            		ds{i} = ['c' num2str(i)];
            	end
            	d.symbols{col} = ds;
            end
              	
            d.data(:,col) = value; 
          
         elseif isempty(checkval(value,'cell','classes ',{size(d,1) 1 0}))
       		col = find(strcmp(d.vartypes,'class'));
            
            if isempty(col),
            	col = size(d,2) + 1;
            	d.vartypes{col} = 'class';
				d.variables{col} = 'class';
				if col == 1,
					d.labels = (1:size(value,1))'
				end
            end
            
            classcodes = zeros(size(value,1),1);
			d.symbols{col} = sortrows(compress(value))';
			for i=1:length(d.symbols{col}),
				classcodes(find(strcmp(value,d.symbols{col}{i}))) = i;
			end
			
			d.data(:,col) = classcodes;
			
	  	end
	  	
      case 'labels',
         error(checkval(value,'real','classes ', ...
            {size(d,1) 1 0}));
         d.labels = value; 
       
      case 'symbols',               
         error(checkval(value,'cell','symbols ', ...
            {1 length(d.symbols) 0}));
         d.symbols=value;
         
      case 'variables'
         error(checkval(value,'cell','variables ', ...
            {1 length(d.variables) 0}));
         d.variables=value;
      
      case 'vartypes'
         error(checkval(value,'cell','vartypes ', ...
            {1 length(d.vartypes) 0}));
         d.vartypes=value;
      
      case 'opt'
		d.opt = value;
      end
      
   else
      error('Arguments mismatch');
   end
end

if nargout == 0,
   assignin('caller',inputname(1),d)
end

