function m = set(m,varargin);
%SET Set object properties
%    
%    SET(M,'ATTRNAME1',ATTRVALUE1,'ATTRNAME2',ATTRVALUE2,...)
%    sets the value of the attribute ATTRNAME1, ATTRNAME2, ...
%    associated to the object M.
%    
%    M_OUT =
%    SET(M,'ATTRNAME1',ATTRVALUE1,'ATTRNAME2',ATTRVALUE2,...)
%    sets the value of the attribute ATTRNAME1, ATTRNAME2, ...
%    associated to the object M and return the modified model
%    inside m_out. The original model is not modified. The
%    following ATTRNAME inherided from the mapping class are
%    recognised:
%    
%      NAME: sets the name of the object
%      N_IN: sets the number of inputs of the model. This
%          sets the saturation level of the inputs to -Inf,
%          Inf.
%      N_OUT: sets the number of outputs of the model
%      USERDATA: sets the 'userData' field
%      OPT: sets the 'opt' field
%      DATE: sets the date of creation of the object
%      LIMITS: sets the limits of the domain of definition
%          of the lazy. The value must be a 2*N_IN matrix
%          where the upper and lever saturation level are
%          defined.
%      OPTIMPARAMS: sets the indices of the parameters to be
%          optimised
%    
%    The following codes specific to the lazy class are
%    recognised:
%    
%      ID_PAR: identification parameters
%      CMB_PAR: model combination parameters
%    
%    The identification parameter can assume the following
%    forms:
%    
%    
%              id_par[3,3]         | idm0  idM0 valM0 |
%                                  | idm1  idM1 valM1 |
%                                  | idm2  idM2 valM2 |
%    
%    
%    where [idmX,idMX] is the range in which the best number
%    of neighbors is searched when identifying the local model
%    of degree X and where valMX is the maximum number of
%    neighbors used in validation for the model of degree X.
%    This means that the model of degree X identified with k
%    neighbors, is validated un the first v neighbors, where
%    v=min(k,valMX).
%    
%      
%              id_par[3,2]         | idm0  idM0 |
%                                  | idm1  idM1 |
%                                  | idm2  idM2 |
%    
%    
%    where idmX and idMX have the same role they have in point
%    1, and valMX is by default set to idMX: each model is
%    validated on all the neighbors used in identification.
%    
%    
%              id_par[3,1]         | c0 |
%                                  | c1 |
%                                  | c2 |
%    
%    
%    Here idmX and idMX are obtained according to the
%    following fotmulas:
%    
%    
%        idmX = 3 * no_parX * cX     idMX = 5 * no_parX * cX
%    
%    
%    where no_parX is the number of parameter of the model of
%    degree X. Recommended choice: cX = 1. As far as the valMX
%    are concerned, they get the default value as in point 2.
%    The default value for cmb_par is 1 as it will be clear
%    from what follows.   If given, cmb_par can assume the
%    following to forms:
%    
%    
%              cmb_par[3,1]        | cmb0 |
%                                  | cmb1 |
%                                  | cmb2 |
%    
%    
%    where cmbX is the number of models of degree X that will
%    be included in the local combination. Each local model
%    will be therfore a combination of "the best cmb0 models
%    of degree 0", "the best cmb1 models of degree 1", and
%    "the best cmb2 models of degree 1" identified as
%    specified by id_par.
%    
%      
%              cmb_par[1,1]        | cmb |
%    
%    
%    where cmb is the number of models that will be combined,
%    disregarding any constraint on the degree of the models
%    that will be considered.  Each local model will be
%    therfore a combination of "the best cmb models",
%    identified as specified by id_par.  
%    
%    

%MAN_PAGE_BEGIN
%
%  @purpose	Set object properties.
%
%  @synopsis		set(m,'AttrName1',AttrValue1,'AttrName2',AttrValue2,...)
%  @description	sets the value of the attribute <code>AttrName1</code>, <code>AttrName2</code>, ...
%		associated to the object <code>m</code>.
%
%  @synopsis		m_out = set(m,'AttrName1',AttrValue1,'AttrName2',AttrValue2,...)
%  @description	sets the value of the attribute <code>AttrName1</code>, <code>AttrName2</code>, ...
%		associated to the object <code>m</code> and return the modified model inside m_out. The original
%		model is not modified.
%
%		The following <CODE>AttrName</CODE> inherided from the
%		mapping class are recognised:
%
%<dl>
%<dt>			name		 <dd>		sets the name of the object
%<dt>			n_in		 <dd>		sets the number of inputs of the model. This
%			sets the saturation level of the inputs to -Inf, Inf.
%<dt>			n_out		 <dd>		sets the number of outputs of the model
%<dt>			userData <dd>		sets the 'userData' field
%<dt>			opt			 <dd>		sets the 'opt' field
%<dt>			date		 <dd>		sets the date of creation of the object
%<dt>	 limits	 <dd>				sets the limits of the domain of definition
%				of the lazy. The value must be a <code>2*n_in</code> matrix where
%			        the upper and lever saturation level are defined.	
%<dt>	 optimparams	 <dd>		sets the indices of the parameters to be optimised
%</dl>
%
%   The following codes specific to the lazy class are recognised:
%<dl>
%<dt>     id_par  <dd>   identification parameters 
%<dt>     cmb_par <dd>   model combination parameters
%</dl>
%The identification parameter can assume the following forms:
%<math>id\_par = [idm idM valM]<alt>
%          id_par[3,3]         | idm0  idM0 valM0 |
%                              | idm1  idM1 valM1 |
%                              | idm2  idM2 valM2 |
%</math>
%                  where [idmX,idMX] is the range in which the best
%                  number of neighbors is searched when identifying
%                  the local model of degree X and where valMX is the
%                  maximum number of neighbors used in validation
%                  for the model of degree X. This means that the
%                  model of degree X identified with k neighbors,
%                  is validated un the first v neighbors, where
%                  v=min(k,valMX).
%<math>id\_par = [idm idM]<alt>  
%          id_par[3,2]         | idm0  idM0 |
%                              | idm1  idM1 |
%                              | idm2  idM2 |
%</math>
%                  where idmX and idMX have the same role they have in
%                  point 1, and valMX is by default set to idMX:
%                  each model is validated on all the neighbors
%                  used in identification.
%<math>id\_par = [c]<alt>
%          id_par[3,1]         | c0 |
%                              | c1 |
%                              | c2 |
%</math>
%                  Here idmX and idMX are obtained according to the
%                  following fotmulas:
%<math>idmX = 3 * no_parX * cX     idMX = 5 * no_parX * cX<alt>
%    idmX = 3 * no_parX * cX     idMX = 5 * no_parX * cX
%</math>
%                  where no_parX is the number of parameter of the model
%                  of degree X. Recommended choice: cX = 1.
%                  As far as the valMX are concerned, they get the
%                  default value as in point 2.
%  
%     The default value for cmb_par is 1 as it will be clear from what
%     follows.   If given, cmb_par can assume the following to forms:
%<math>cmb\_par = [cmb]<alt>
%          cmb_par[3,1]        | cmb0 |
%                              | cmb1 |
%                              | cmb2 |
%</math>
%                  where cmbX is the number of models of degree X
%                  that will be included in the local combination.
%                  Each local model will be therfore a combination of
%                  "the best cmb0 models of degree 0",
%                  "the best cmb1 models of degree 1", and
%                  "the best cmb2 models of degree 1" identified
%                  as specified by id_par.
%<math>cmb_par = [cmb]<alt>  
%          cmb_par[1,1]        | cmb |
%</math>
%                  where cmb is the number of models that will be
%                  combined, disregarding any constraint on the degree
%                  of the models that will be considered.  Each local
%                  model will be therfore a combination of "the best
%                  cmb models", identified as specified by id_par.  
%  
%MAN_PAGE_END		


% Check that we have the right number of inputs
ni = nargin;
if (ni<3)|(ni/2 == floor(ni/2)),
  error('Wrong number of inputs'); %
end

if ~isa(m,'lazy'),
  % Call built-in SET. Handles calls like set(gcf,'user',lazy)	
  builtin('set',m,varargin{:});
  return
end

% For each couple of inputs
for i=1:(ni - 1)/2,
  name = varargin{i*2-1};
  value = varargin{i*2};
	
		
  switch name,
	  
   case 'id_par'
    if (iscompat(value, ones(3,1))) & min(value,1)>= 0,
      m.id_par=value;
    elseif (iscompat(value, ones(3,2))) & min(value(:,2)-value(:,1),1)>= 0,
      m.id_par=value;
    elseif (iscompat(value, ones(3,3))) & min(value(:,2)-value(:,1),1)>= 0,
      m.id_par=value;
    else  
      error('incorrect value or format of the ''id_par'' parameter');
    end
    
   case 'cmb_par'
    if (iscompat(value, ones(1,1))) & min(value,1)>= 0,
      m.cmb_par=value;
    elseif (iscompat(value, ones(3,1))) & min(value,1)>= 0,
      m.cmb_par=value;
    else  
      error('incorrect value or format of the ''cmb_par'' parameter');
    end
    
   case 'examples_y' 
    m.examples_y=value;
    
   case 'examples_x' 
    m.examples_x=value; 
	 
   otherwise
    m.mapping = set(m.mapping, name, value);
    
  end
  
end
	

if nargout == 0,
	assignin('caller',inputname(1),m)
end

