function m = system(varargin)
%SYSTEM Constructor for the system abstract class
%    
%    M=SYSTEM creates a new general system skeleton
%    
%    M=SYSTEM(NAME,N_IN,N_OUT) creates a new general system
%    with N_IN inputs and N_OUT outputs. The name NAME is
%    associated to it.
%    
%    The class defines the following methods:
%    
%      ADD_DATA: add a input-output data set to the model
%      ADD_MAPPING: add a mapping to a system object
%      CLONE_OUTPUT: clone the mapping associated to an
%          output
%      DENORMALISE: normalises the system M
%      DISPLAY: Display an object of class system
%      EVAL: computes the value of the system for some
%          regressor
%      GET: gets the value of the attribute of an object
%      IDENTIFY: Identify the system from data (abstact
%          method)
%      INTERP_MODEL: computes the jacobian of the model by
%          interpoling local models
%      JACOB_INPUTS: computes the jacobian of the model
%      JACOB_PARAMS: computes the jacobian of the model
%      NORMALISE: normalises the system M
%      PLOT: plot the system
%      SET: Set object properties
%    
%    See also: 
%    
%    

%MAN_PAGE_BEGIN
%
%   @purpose	Constructor for the system abstract class
%
%   @synopsis	m=system
%   @description  creates a new general system skeleton
%
%   @synopsis	m=system(name,n_in,n_out)
%   @description  creates a new general system with <CODE>n_in</CODE> inputs
%	and <CODE>n_out</CODE> outputs. The name <CODE>name</CODE> is associated to it.
%	
%
%   @see 
%
%MAN_PAGE_END

error(checkargs(varargin,[0 3],'char','numeric','numeric'));

%Some note about the input/output representation: mapping is a cell array containing the
%input output static mapping for every output. links is the field that make the connection
%between each output of the system and each output of the mapping(s)

m=[];
if nargin == 0,
	m.name = 'untitled system';
	m.n_in = 0;                       %number of inputs 
	m.n_out = 0;                      %number of outputs
	m.data = [];
	m.T = -1;
	m.mapping = {};
	m.links = [];
	m.limits = [];
	m.userData = [];
	m.opt = [];
	m.date = clock;                 %creation date
	m = class(m,'system');
elseif nargin == 1,
	m.name = varargin{1};
	m.n_in=1;                       %number of inputs 
	m.n_out=1;                      %number of outputs
	m.data = [];
	m.T = -1;
	m.mapping = {linear([varargin{1} ' linear mapping'],1,1)};
	m.links = [1 1];
	m.limits=[-inf -inf;inf inf];
	m.userData=[];
	m.opt=[];
	m.date = clock;                 %creation date
	m=class(m,'system');
elseif nargin == 2,
	m.name = varargin{1};
	m.n_in = varargin{2};           %number of inputs 
	m.n_out = 1;                      %number of outputs
	m.data = [];
	m.T = -1;
	m.mapping = {linear([varargin{1} ' linear mapping'],varargin{2},1)};
	m.links = [1 1];
	m.limits=[[-inf;inf] * ones(1,varargin{2}) [-inf; inf]];
	m.userData=[];
	m.opt=[];
	m.date = clock;                 %creation date
	m=class(m,'system');
elseif nargin == 3,
	m.name = varargin{1};
	m.n_in = varargin{2};           %number of inputs 
	m.n_out = varargin{3};                      %number of outputs
	m.data = [];
	m.T = -1;
	m.mapping = {linear([varargin{1} ' linear mapping'],varargin{2},varargin{3})};
	m.links = [ones(varargin{3},1) (1:varargin{3})'];
	m.limits=[[-inf;inf] * ones(1,varargin{2}) [-inf; inf] * ones(1,varargin{3})];
	m.userData=[];
	m.opt=[];
	m.date = clock;                 %creation date
	m=class(m,'system');
end
	
	
